let audioUrls = [];

// 初始化：从存储加载音频列表
chrome.storage.local.get(['audioUrls'], (result) => {
  if (result.audioUrls) {
    audioUrls = result.audioUrls;
    updateBadgeCount(); // 初始化徽章计数
  }
});

// 更新徽章计数
function updateBadgeCount() {
  const count = audioUrls.length;
  if (count > 0) {
    // 设置徽章文本为记录数，颜色为白色，背景为蓝色
    chrome.action.setBadgeText({ text: count.toString() });
    chrome.action.setBadgeBackgroundColor({ color: '#4285f4' }); // 蓝色背景
    chrome.action.setBadgeTextColor({ color: '#ffffff' }); // 白色文本
  } else {
    // 计数为0时清空徽章
    chrome.action.setBadgeText({ text: '' });
  }
}

// 定义常见音频文件扩展名
const AUDIO_EXTENSIONS = ['mp3', 'wav', 'flac', 'ogg', 'aac', 'm4a', 'wma'];

// 从字符串（URL 或文件名）中提取扩展名并判断是否为音频
function isAudioByExtension(str) {
  if (!str) return false;
  // 提取最后一个 "." 后的扩展名（忽略大小写）
  const ext = str.split('.').pop().toLowerCase();
  return AUDIO_EXTENSIONS.includes(ext);
}

// 监听网络请求，捕获音频文件
chrome.webRequest.onCompleted.addListener(
  (details) => {
    // 检查响应头中的Content-Type是否为音频类型
    const contentTypeHeader = details.responseHeaders.find(
      header => header.name.toLowerCase() === 'content-type'
    );
	
	let isAudio = false;
	
	if (contentTypeHeader && contentTypeHeader.value.startsWith('audio/')) {
		isAudio = true;
    }else if (contentTypeHeader && contentTypeHeader.value === 'application/octet-stream') {
		// 检查 URL 中的扩展名
		const isUrlAudio = isAudioByExtension(details.url);

		// 检查 Content-Disposition 中的文件名扩展名
		const dispositionHeader = details.responseHeaders.find(
			header => header.name.toLowerCase() === 'content-disposition'
		);
		
		const fileName = dispositionHeader?.value.match(/filename="?([^"]+)"?/)?.[1]; // 提取文件名
		const isDispositionAudio = isAudioByExtension(fileName);

		// 只要 URL 或文件名有一个符合音频扩展名，就视为音频
		isAudio = isUrlAudio || isDispositionAudio;
	}
	
	if(isAudio){
		if (!audioUrls.includes(details.url)) {
			audioUrls.push(details.url);
			// 保存到本地存储
			chrome.storage.local.set({ audioUrls: audioUrls });
			// 更新徽章计数
			updateBadgeCount();
			// 通知popup更新列表
			chrome.runtime.sendMessage({
				action: 'updateList',
				audioUrls: audioUrls
			}).catch(() => {
				// 忽略popup未打开时的错误
			});
		}
	}
  },
  { urls: ["<all_urls>"] },
  ["responseHeaders"]
);

// 处理来自popup的消息
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  switch (message.action) {
    case 'getAudioList':
      sendResponse({ audioUrls: audioUrls });
      break;
      
    case 'removeAudio':
      if (message.url) {
        audioUrls = audioUrls.filter(url => url !== message.url);
        chrome.storage.local.set({ audioUrls: audioUrls });
        // 更新徽章计数
        updateBadgeCount();
        // 返回更新后的列表
        sendResponse({ success: true, updatedList: audioUrls });
        // 通知popup更新
        chrome.runtime.sendMessage({
          action: 'updateList',
          audioUrls: audioUrls
        }).catch(() => {
          // 忽略popup未打开时的错误
        });
      }
      break;
      
    case 'clearAll':
      audioUrls = [];
      chrome.storage.local.set({ audioUrls: audioUrls });
      // 更新徽章计数
      updateBadgeCount();
      // 返回空列表
      sendResponse({ success: true, updatedList: audioUrls });
      // 通知popup更新
      chrome.runtime.sendMessage({
        action: 'updateList',
        audioUrls: audioUrls
      }).catch(() => {
        // 忽略popup未打开时的错误
      });
      break;
      
    case 'download':
		if (message.url && message.blobUrl) {
			const fileName = message.url.split('/').pop().split('?')[0];
		  
			chrome.downloads.download({
				url: message.blobUrl,
				filename: fileName,
				saveAs: false
			}, (downloadId) => {
				if (chrome.runtime.lastError) {
					//下载失败时返回错误信息
					sendResponse({ 
						success: false, 
						error: chrome.runtime.lastError.message 
					});
				} else {
					sendResponse({ success: true });
				}
			});

			//异步响应需要返回true（先return true告诉 Chrome 保留消息通道，等待异步 sendResponse 被调用。没有return true通道会提前关闭，导致响应无法送达。）
			return true;
		}
	break;
  }
});
    